/* batch_score_animation_3.jsx — stroke routing + solid recolor + precomp-first + purge (no logs)

CSV columns (required): abbreviation,R,G,B,R2,G2,B2
Optional: league,displayName

ENV you can set:
  AE_PROJECT, AE_CSV, AE_COMP
  AE_TEXT_STROKE_LAYER="TOUCHDOWN"
  AE_TD_PRECOMP_LAYER="TD PreComp"
  AE_TEAMNAME_LAYER="TeamName"
  AE_SOLID_LAYER="Solid"                // <-- solid to recolor to PRIMARY

  AE_STROKE_WIDTH=""                    // "" = don't touch widths; otherwise "2", etc.
  AE_STROKE_OVER="1|0"

  AE_LOGO_LAYER="Logo"
  AE_LOGO_DIR="/path/to/Logos"
  AE_LOGO_PATH_TEMPLATE="{league}/{abbr}"
  AE_LOGO_EXTS="png,jpg,jpeg,svg,ai,psd"
  AE_LOGO_CASE="keep|upper|lower"

  AE_LEAGUE, AE_TEAMS, AE_LIMIT
  AE_OUTDIR, AE_PATH_TEMPLATE, AE_ANIM, AE_RS_TEMPLATE, AE_OM_TEMPLATE, AE_EXT
  AE_PURGE_BEFORE_RENDER="1|0"          // purge caches before render (default 1)
  AE_NO_RENDER="0|1"
  AE_QUIT="1"
*/

(function () {
  // ---------- tiny utils ----------
  function env(k,d){ var v=$.getenv(k); return (v===null||v===undefined||v==="")?d:v; }
  function fail(msg){ alert(msg); throw new Error(msg); }
  function trim(s){ var t=String(s||""); while(/^[ \t\r\n]/.test(t)) t=t.substring(1); while(/[ \t\r\n]$/.test(t)) t=t.substring(0,t.length-1); return t; }
  function toLower(s){ return String(s||"").toLowerCase(); }
  function clamp01(v){ return Math.max(0, Math.min(1, v)); }
  function sanitize(s){ var bad='\\/:*?"<>|', t=String(s||""), out="", i, ch; for(i=0;i<t.length;i++){ ch=t.charAt(i); out+=(bad.indexOf(ch)>=0?"-":ch);} return out; }
  function hasExt(p){ return /\.[A-Za-z0-9]+$/.test(p); }

  function numOr(v,def){ var n=parseFloat(v); return (isFinite(n) ? n : def); }
  function safeWidth(w){ w=numOr(w,2); if (w<=0) w=0.1; return w; }
  function safeColor(c){
    if (!c || c.length!==3) return [0,0,0];
    function cl(x){ return (x<0?0:(x>1?1:x)); }
    var r=numOr(c[0],0), g=numOr(c[1],0), b=numOr(c[2],0);
    return [cl(r),cl(g),cl(b)];
  }
  // --- add below safeColor/rgb01 ---
  function isNearlyBlack(c){
    if (!c || c.length!==3) return true;
    return (c[0] <= 0.05 && c[1] <= 0.05 && c[2] <= 0.05); // threshold ~13/255
  }
  function ensureNonBlack(preferred, alternate){
    var p = safeColor(preferred);
    var a = safeColor(alternate);
    if (!isNearlyBlack(p)) return p;
    if (!isNearlyBlack(a)) return a;
    return [1,1,1]; // both black → use white
  }
  function darkenColor(c, factor){
    var f = numOr(factor, 0.7); // 30% less bright = x0.7
    var s = safeColor(c);
    return [clamp01(s[0]*f), clamp01(s[1]*f), clamp01(s[2]*f)];
  }

  function rgb01(r,g,b){ return [clamp01(numOr(r,0)/255), clamp01(numOr(g,0)/255), clamp01(numOr(b,0)/255)]; }
  function trySet(p,v){ if (!p) return false; try{ p.setValue(v); return true; }catch(e){ return false; } }

  function unhideAll(comp){
    try{ comp.hideShyLayers=false; }catch(e){}
    for (var i=1;i<=comp.numLayers;i++){
      var L=comp.layer(i);
      try{ L.enabled=true; }catch(e){}
      try{ L.shy=false; }catch(e){}
      try{ if (L.source && (L.source instanceof CompItem)) unhideAll(L.source); }catch(e){}
    }
  }

  // ---------- env ----------
  var PROJECT   = env("AE_PROJECT", null);
  var CSV_PATH  = env("AE_CSV", null);
  var COMP_NAME = env("AE_COMP","Comp 1");

  var TEXT_STROKE_LAYER = env("AE_TEXT_STROKE_LAYER","TOUCHDOWN");
  var TD_PRECOMP_LAYER  = env("AE_TD_PRECOMP_LAYER","TD PreComp");
  var TEAMNAME_LAYER    = env("AE_TEAMNAME_LAYER","TeamName");
  var SOLID_LAYER       = env("AE_SOLID_LAYER","Solid");

  var STROKE_W  = env("AE_STROKE_WIDTH","");
  var STROKE_OV = env("AE_STROKE_OVER","");

  var LOGO_LAYER = env("AE_LOGO_LAYER","Logo");
  var LOGO_DIR   = env("AE_LOGO_DIR","");
  var LOGO_TPL   = env("AE_LOGO_PATH_TEMPLATE","{league}/{abbr}");
  var LOGO_EXTS  = env("AE_LOGO_EXTS","png,jpg,jpeg,svg,ai,psd").split(",");
  var LOGO_CASE  = env("AE_LOGO_CASE","keep");

  var LEAGUE    = env("AE_LEAGUE","");
  var TEAMS_LIST= env("AE_TEAMS","");
  var LIMIT_STR = env("AE_LIMIT","");
  var LIMIT     = (LIMIT_STR && !isNaN(parseInt(LIMIT_STR,10))) ? parseInt(LIMIT_STR,10) : null;

  var OUTDIR    = env("AE_OUTDIR","");
  var PATH_TPL  = env("AE_PATH_TEMPLATE","{league}/{abbr}");
  var ANIM_NAME = env("AE_ANIM","TOUCHDOWN");
  var RS_TPL    = env("AE_RS_TEMPLATE","Best Settings");
  var OM_TPL    = env("AE_OM_TEMPLATE","H.264 - Match Source - High Bitrate");
  var EXT       = env("AE_EXT",".mp4");
  var PURGE     = (env("AE_PURGE_BEFORE_RENDER","1")==="1");
  var NO_RENDER = (env("AE_NO_RENDER","0")==="1");
  var QUIT_APP  = (env("AE_QUIT","1")==="1");

  // ---------- CSV ----------
  function openRead(path){ var f=new File(path); if(!f.exists) fail("File not found: "+path); f.open("r"); var s=f.read(); f.close(); return s; }
  function parseCSV(txt){
    var raw=txt.split(/\r\n|\n|\r/), lines=[], i; for(i=0;i<raw.length;i++){ var L=raw[i]; if(L && !/^\s*$/.test(L)) lines.push(L); }
    var rows=[], c; for(i=0;i<lines.length;i++){ var line=lines[i]; var cells=(line.indexOf("\t")!==-1? line.split("\t") : line.split(",")); 
      for(c=0;c<cells.length;c++){ var cell=trim(cells[c]); if(cell.charAt(0)=='"' && cell.charAt(cell.length-1)=='"') cell=cell.substring(1,cell.length-1); cells[c]=cell; }
      rows.push(cells);
    }
    if(rows.length<2) fail("CSV has no data."); return rows;
  }
  function headerIdx(h){
    var m={}, i; for(i=0;i<h.length;i++) m[toLower(h[i])]=i;
    function need(x){ if(m[x]===undefined) fail("Missing column: "+x); }
    need("abbreviation"); need("r"); need("g"); need("b"); need("r2"); need("g2"); need("b2");
    return m;
  }
  function buildTeams(rows){
    var h=rows[0], idx=headerIdx(h), out=[], i, hasLeague=(idx["league"]!==undefined), hasName=(idx["displayname"]!==undefined);
    for(i=1;i<rows.length;i++){
      var r=rows[i], ab=r[idx["abbreviation"]]; if(!ab) continue;
      out.push({
        abbr: r[idx["abbreviation"]],
        league: hasLeague ? r[idx["league"]] : (LEAGUE||"NA"),
        name: hasName ? r[idx["displayname"]] : r[idx["abbreviation"]],
        primary:   rgb01(r[idx["r"]],  r[idx["g"]],  r[idx["b"]]),
        secondary: rgb01(r[idx["r2"]], r[idx["g2"]], r[idx["b2"]])
      });
    }
    return out;
  }
  function pickTeams(all){
    var res=[], i;
    if (TEAMS_LIST){ var want={}, parts=TEAMS_LIST.split(","); for(i=0;i<parts.length;i++){ want[trim(parts[i])] = true; } for(i=0;i<all.length;i++){ if(want[all[i].abbr]) res.push(all[i]); } }
    else if (LEAGUE){ for(i=0;i<all.length;i++){ if(toLower(all[i].league)===toLower(LEAGUE)) res.push(all[i]); } }
    else { res = all.slice(0); }
    if (LIMIT && res.length>LIMIT) res = res.slice(0, LIMIT);
    return res;
  }

  // ---------- AE helpers ----------
  function findComp(name){ for(var i=1;i<=app.project.numItems;i++){ var it=app.project.item(i); if(it instanceof CompItem && it.name===name) return it; } return null; }
  function getLayer(comp, name){ var ly = comp.layer(name); if(!ly) return null; return ly; }

  // Source Text stroke only
  function setTextStrokeOnly(layer, color, widthOpt, overOpt){
    var st=layer.property("Source Text"); if(!st) return false;
    var td=st.value;
    td.applyStroke = true;
    td.strokeColor = safeColor(color);
    if (overOpt!==""){ td.strokeOverFill = (overOpt==="1" || overOpt===true); }
    if (widthOpt!==""){ td.strokeWidth = safeWidth(widthOpt); }
    try{ st.setValue(td); return true; }catch(e){ return false; }
  }

  // Text Animator -> Stroke Color
  function setTextAnimatorStroke(layer, color){
    var t = layer.property("Text"); if(!t) return 0;
    var anims = t.property("Animators"); if(!anims) return 0;
    var n=0;
    for (var i=1;i<=anims.numProperties;i++){
      var a = anims.property(i);
      var props = a.property("ADBE Text Animator Properties");
      if (props){
        var sc = props.property("ADBE Text Stroke Color");
        if (sc) { if (trySet(sc, safeColor(color))) n++; }
      }
    }
    return n;
  }

  // TeamName: no fill + stroke secondary
  function setTeamNameStrokeOnly(layer, teamName, strokeColor, widthOpt){
    var st=layer.property("Source Text"); if(!st) return false;
    var td=st.value;
    td.text = String(teamName||"");
    td.applyFill = false;
    td.applyStroke = true;
    td.strokeColor = safeColor(strokeColor);
    if (widthOpt!==""){ td.strokeWidth = safeWidth(widthOpt); }
    td.strokeOverFill = true;
    try{ st.setValue(td); }catch(e){ return false; }
    setTextAnimatorStroke(layer, strokeColor);
    return true;
  }

  // Effects/Layer Styles/Shape strokes (stroke only)
  function setEffectsStrokeOnly(layer, color, widthOpt){
    var fx = layer.property("Effects"); if(!fx) return 0;
    var n=0;
    for (var i=1;i<=fx.numProperties;i++){
      var eff = fx.property(i);
      if (eff.matchName==="ADBE Stroke"){
        if (trySet(eff.property("Color"), safeColor(color))) n++;
        if (widthOpt!==""){ var bs=eff.property("Brush Size"); if (bs) { if (trySet(bs, safeWidth(widthOpt))) n++; } }
      }
    }
    return n;
  }
  function setLayerStylesStrokeOnly(layer, color, widthOpt){
    var ls = layer.property("Layer Styles"); if(!ls) return 0;
    var n=0;
    var stroke = ls.property("Stroke");
    if (stroke){
      if (trySet(stroke.property("Color"), safeColor(color))) n++;
      if (widthOpt!==""){ var s=stroke.property("Size"); if (s) { if (trySet(s, safeWidth(widthOpt))) n++; } }
    }
    return n;
  }
  function setShapeStrokesOnly(shapeLayer, color, widthOpt){
    if (shapeLayer.matchName!=="ADBE Vector Layer") return 0;
    var contents = shapeLayer.property("Contents"); if(!contents) return 0;
    var changed=0;
    function walk(pg){
      for (var i=1;i<=pg.numProperties;i++){
        var p = pg.property(i), mn = p.matchName;
        if (mn==="ADBE Vector Group" || mn==="ADBE Vector Filter Group" || mn==="ADBE Vector Shape - Group"){
          var sub = p.property("Contents"); if (sub) walk(sub);
        } else if (mn==="ADBE Vector Graphic - Stroke"){
          var c=null,w=null;
          for (var j=1;j<=p.numProperties;j++){
            var x=p.property(j);
            if (x.matchName==="ADBE Vector Stroke Color") c=x;
            if (x.matchName==="ADBE Vector Stroke Width") w=x;
          }
          if (c) { if (trySet(c, safeColor(color))) changed++; }
          if (w && widthOpt!==""){ if (trySet(w, safeWidth(widthOpt))) changed++; }
        }
      }
    }
    walk(contents);
    return changed;
  }

  // Solid recolor (to PRIMARY)
  function setSolidPrimaryFill(layer, color){
    // 1) True solid
    try{
      var src = layer.source;
      if (src && src.mainSource && src.mainSource.color !== undefined) {
        src.mainSource.color = safeColor(color);
        return true;
      }
    }catch(e){}
    // 2) Fill effect
    var fx = layer.property("Effects");
    if (fx){
      var fill = fx.property("Fill") || fx.addProperty("ADBE Fill");
      if (fill){ return trySet(fill.property("Color"), safeColor(color)); }
    }
    // 3) Layer Styles / Color Overlay or Fill
    var ls = layer.property("Layer Styles");
    if (ls){
      var overlay = ls.property("Color Overlay");
      if (overlay){ if (trySet(overlay.property("Color"), safeColor(color))) return true; }
      var fillLS = ls.property("Fill");
      if (fillLS){ if (trySet(fillLS.property("Color"), safeColor(color))) return true; }
    }
    return false;
  }

  // Precomp walker: set strokes everywhere that could matter
  function strokePrecompEverything(comp, color, widthOpt, overOpt){
    var n=0;
    for (var i=1;i<=comp.numLayers;i++){
      var L=comp.layer(i);
      if (L.matchName==="ADBE Text Layer"){
        if (setTextStrokeOnly(L, color, widthOpt, overOpt)) n++;
        n += setTextAnimatorStroke(L, color);
      }
      n += setEffectsStrokeOnly(L, color, widthOpt);
      n += setLayerStylesStrokeOnly(L, color, widthOpt);
      n += setShapeStrokesOnly(L, color, widthOpt);
      try{
        if (L.source && (L.source instanceof CompItem)){
          n += strokePrecompEverything(L.source, color, widthOpt, overOpt);
        }
      }catch(e){}
    }
    return n;
  }

  // Logos
  function applyCase(s){
    if (LOGO_CASE==="upper") return String(s||"").toUpperCase();
    if (LOGO_CASE==="lower") return String(s||"").toLowerCase();
    return String(s||"");
  }
  function findLogoFile(league, abbr){
    if (!LOGO_DIR) return null;
    var L = applyCase(league||"");
    var A = applyCase(abbr||"");
    var tpl = LOGO_TPL.replace("{league}",L).replace("{abbr}",A)
                      .replace("{league_lc}", String(league||"").toLowerCase())
                      .replace("{abbr_lc}",   String(abbr||"").toLowerCase())
                      .replace("{league_uc}", String(league||"").toUpperCase())
                      .replace("{abbr_uc}",   String(abbr||"").toUpperCase());
    if (hasExt(tpl)){
      var f1 = new File(LOGO_DIR + "/" + tpl);
      if (f1.exists) return f1; else return null;
    }
    for (var i=0;i<LOGO_EXTS.length;i++){
      var f2 = new File(LOGO_DIR + "/" + tpl + "." + trim(LOGO_EXTS[i]));
      if (f2.exists) return f2;
    }
    return null;
  }
  function replaceLogo(comp, layerName, league, abbr){
    var lyr = comp.layer(layerName); if (!lyr || !LOGO_DIR) return;
    var f = findLogoFile(league, abbr); if (!f) return;
    var io = new ImportOptions(f); if (!io.canImportAs(ImportAsType.FOOTAGE)) return;
    var footage = app.project.importFile(io);
    lyr.replaceSource(footage, false);
  }

  // ---------- run ----------
  if (app.beginSuppressDialogs){ try{ app.beginSuppressDialogs(); }catch(e){} } // no args for max compat
  app.beginUndoGroup("Stroke + Solid + Logo + Render");

  if(!PROJECT) fail("AE_PROJECT env not set.");
  var aep=new File(PROJECT); if(!aep.exists) fail("AE_PROJECT not found: "+PROJECT);
  app.open(aep);

  if(!CSV_PATH) fail("AE_CSV env not set.");
  var rows=parseCSV(openRead(CSV_PATH)), teams=buildTeams(rows), todo=pickTeams(teams);
  if(!todo.length) fail("No teams matched.");

  var rootComp=findComp(COMP_NAME); if(!rootComp) fail("Comp not found: "+COMP_NAME);
  unhideAll(rootComp);

  var rootOut = OUTDIR ? new Folder(OUTDIR) : (app.project.file ? app.project.file.parent : Folder.desktop);
  if(!rootOut.exists) rootOut.create();

  for (var i=0;i<todo.length;i++){
    var t = todo[i];
    // choose non-black variants and precompute the darkened secondary for TeamName
    var basePrimary   = safeColor(t.primary);
    var baseSecondary = safeColor(t.secondary);

    var C_PRIMARY   = ensureNonBlack(basePrimary, baseSecondary);
    var C_SECONDARY = ensureNonBlack(baseSecondary, basePrimary);
    var C_TEAMNAME  = darkenColor(C_SECONDARY, 0.7); // 30% less bright for TeamName


    // --- 1) change strokes in root + precomp SOURCE (before render)
    var tdLayer = getLayer(rootComp, TEXT_STROKE_LAYER);
    if (tdLayer){ setTextStrokeOnly(tdLayer, C_PRIMARY, STROKE_W, STROKE_OV); setTextAnimatorStroke(tdLayer, C_PRIMARY); }

    var preLayer = getLayer(rootComp, TD_PRECOMP_LAYER);
    if (preLayer && preLayer.source && (preLayer.source instanceof CompItem)){
      strokePrecompEverything(preLayer.source, C_PRIMARY, STROKE_W, STROKE_OV);
    }

    var nameLy = getLayer(rootComp, TEAMNAME_LAYER);
    if (nameLy){ setTeamNameStrokeOnly(nameLy, t.name, C_TEAMNAME, STROKE_W); }


    // Solid = primary
    var solidLy = getLayer(rootComp, SOLID_LAYER);
    if (solidLy){ setSolidPrimaryFill(solidLy, C_PRIMARY); }

    // Logo
    replaceLogo(rootComp, LOGO_LAYER, t.league, t.abbr);

    // Optional purge to avoid stale caches
    if (PURGE && app.purge) {
      try { app.purge(PurgeTarget.ALL_CACHES); } catch(e) {}
    }

    // --- 2) now render
    if (!NO_RENDER){
      var leagueLabel=t.league||LEAGUE||"NA";
      var sub = PATH_TPL.replace("{league}",sanitize(leagueLabel)).replace("{abbr}",sanitize(t.abbr));
      var outFolder=new Folder(rootOut.fsName+"/"+sub); if(!outFolder.exists) outFolder.create();
      var outFile=File(outFolder.fsName+"/"+sanitize(leagueLabel)+"_"+sanitize(leagueLabel)+"_"+sanitize(t.abbr)+"_"+sanitize(ANIM_NAME)+EXT);

      var rqItem = app.project.renderQueue.items.add(rootComp);
      try{ rqItem.applyTemplate(RS_TPL); }catch(e){}
      var om=rqItem.outputModule(1);
      try{ om.applyTemplate(OM_TPL); }catch(e){}
      om.file = outFile;
      app.project.renderQueue.render();
    }
  }

  app.endUndoGroup();
  if (app.endSuppressDialogs){ try{ app.endSuppressDialogs(); }catch(e){} }
  if (QUIT_APP) app.quit();
})();
